/*************************************************************************
 * The contents of this file are subject to the MYRICOM MYRINET          *
 * EXPRESS (MX) NETWORKING SOFTWARE AND DOCUMENTATION LICENSE (the       *
 * "License"); User may not use this file except in compliance with the  *
 * License.  The full text of the License can found in LICENSE.TXT       *
 *                                                                       *
 * Software distributed under the License is distributed on an "AS IS"   *
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied.  See  *
 * the License for the specific language governing rights and            *
 * limitations under the License.                                        *
 *                                                                       *
 * Copyright 2003 - 2004 by Myricom, Inc.  All rights reserved.          *
 *************************************************************************/

static const char __idstring[] = "@(#)$Id: mx_mcp_wrapper_common.c,v 1.117 2006/05/29 03:52:15 loic Exp $";

#include "mx_arch.h"
#include "mcp_global.h"
#include "mcp_config.h"
#define MX_MCP_INTERFACE_INTERNALS 1
#include "mx_mcp_interface.h"
#include "mx_pio.h"

/* parameters which should be udated by the driver before it asks us
   to load our mcp */

mcp_select_mcp_t mx_mcp_select_mcp;
mcp_set_globals_t mx_mcp_set_globals;
mcp_set_param_t mx_mcp_set_param;
mcp_get_param_t mx_mcp_get_param;
mcp_get_counters_t mx_mcp_get_counters;
mcp_get_log_events_t mx_mcp_get_log_events;
mcp_get_log_events_count_t mx_mcp_get_log_events_count;
mcp_init_mcp_t mx_mcp_init_mcp;

/* variables exported from ../mcp/mx_mcp_array.c */
extern unsigned char mx_mcp_array_d[];
extern unsigned char mx_mcp_array_e[];
extern unsigned char mx_mcp_array_z[];
extern uint32_t mx_mcp_array_d_length;
extern uint32_t mx_mcp_array_e_length;
extern uint32_t mx_mcp_array_z_length;
extern const char *mx_mcp_counters_d[];
extern const char *mx_mcp_counters_e[];
extern const char *mx_mcp_counters_z[];
extern int mx_mcp_counters_d_count;
extern int mx_mcp_counters_e_count;
extern int mx_mcp_counters_z_count;
extern const char *mx_mcp_d_log_events[];
extern const char *mx_mcp_z_log_events[];
extern int mx_mcp_d_log_events_count;
extern int mx_mcp_z_log_events_count;
extern const unsigned int mx_mcp_d_parity_critical_start;
extern const unsigned int mx_mcp_d_parity_critical_end;
extern const unsigned int mx_mcp_e_parity_critical_start;
extern const unsigned int mx_mcp_e_parity_critical_end;

mcp_interface_t mx_mcpi = {
  /* version */             MX_MCP_INTERFACE_VERSION,
  /* memory_size */         sizeof(mx_mcp_public_global_t),
  /* globals */             0,
  /* init_mcp */            mx_mcp_init_mcp,
  /* mcp_select_mcp_t */    mx_mcp_select_mcp,
  /* mcp_set_globals_t */   mx_mcp_set_globals,
  /* mcp_set_param_t */     mx_mcp_set_param,
  /* mcp_get_param_t */     mx_mcp_get_param,
  /* mcp_get_counters_t */  mx_mcp_get_counters,
  /* mcp_get_log_events_t */  mx_mcp_get_log_events,
  /* mcp_get_log_events_count_t */  mx_mcp_get_log_events_count
};

#define Globals(x) mx_mcpi.globals[(x)]

int
mx_mcp_init_mcp(int unit, int board_type)
{
  switch (board_type) {
  case MX_BOARD_TYPE_D:
  case MX_BOARD_TYPE_Z:
    Globals(unit).driver_api_version = MX_MCP_DRIVER_API_VERSION;
    Globals(unit).mcp_version = MX_MCP_VERSION;
    Globals(unit).nodes_cnt = MX_MCP_NODES_CNT;
    Globals(unit).endpoints_cnt = MX_MCP_ENDPOINTS_CNT;
    Globals(unit).send_handles_cnt = MX_MCP_SEND_HANDLES_CNT;
    Globals(unit).pull_handles_cnt = MX_MCP_PULL_HANDLES_CNT;
    Globals(unit).push_handles_cnt = MX_MCP_PUSH_HANDLES_CNT;
    Globals(unit).rdma_windows_cnt = MX_MCP_RDMA_WINDOWS_CNT;
    Globals(unit).peer_hash_size = MX_MCP_PEER_HASH;
    Globals(unit).intr_coal_delay = MX_MCP_INTR_COAL_DELAY;
    break;

  case MX_BOARD_TYPE_E:
    Globals(unit).driver_api_version = MX_MCP_DRIVER_API_VERSION;
    Globals(unit).mcp_version = MX_MCP_VERSION;
    Globals(unit).nodes_cnt = MX_MCP_NODES_CNT;
    Globals(unit).endpoints_cnt = MX_MCP_ENDPOINTS_CNT;
    Globals(unit).send_handles_cnt = MX_MCP_SEND_HANDLES_CNT;
    Globals(unit).pull_handles_cnt = MX_MCP_PULL_HANDLES_CNT;
    Globals(unit).push_handles_cnt = MX_MCP_PUSH_HANDLES_CNT;
    Globals(unit).rdma_windows_cnt = MX_MCP_RDMA_WINDOWS_CNT;
    Globals(unit).peer_hash_size = MX_MCP_PEER_HASH;
    Globals(unit).intr_coal_delay = MX_MCP_INTR_COAL_DELAY;
    break;
  
  default:
    /* we should never get here */
    return 1;
  }
  
  return 0;
}

int
mx_mcp_get_counters(int board_type, const char ***counters, uint32_t *counters_count)
{
  switch (board_type) {
#if MX_2G_ENABLED  
  case MX_BOARD_TYPE_D:
    *counters = mx_mcp_counters_d;
    *counters_count = mx_mcp_counters_d_count;
    break;

  case MX_BOARD_TYPE_E:
    *counters = mx_mcp_counters_e;
    *counters_count = mx_mcp_counters_e_count;
    break;
#endif

#if MX_10G_ENABLED  
  case MX_BOARD_TYPE_Z:
    *counters = mx_mcp_counters_z;
    *counters_count = mx_mcp_counters_z_count;
    break;
#endif
  
  default:
    /* we should never get here */
    return 1;
  }
  return 0;
}

void *
mx_mcp_get_log_events(int board_type)
{
  switch (board_type) {
#if MX_10G_ENABLED
  case MX_BOARD_TYPE_Z:
    return (void*)mx_mcp_z_log_events;
#endif
#if MX_2G_ENABLED
  case MX_BOARD_TYPE_E:
  case MX_BOARD_TYPE_D:
    return (void*)mx_mcp_d_log_events;
#endif
  }
  return NULL;
}

int
mx_mcp_get_log_events_count(int board_type)
{
  switch (board_type) {
#if MX_10G_ENABLED
  case MX_BOARD_TYPE_Z:
    return mx_mcp_z_log_events_count;
#endif
#if MX_2G_ENABLED
  case MX_BOARD_TYPE_E:
  case MX_BOARD_TYPE_D:
    return mx_mcp_d_log_events_count;
#endif
  }
  return 0;
}

int
mx_mcp_select_mcp(int unit, int board_type, unsigned char **mcp,
		  int *len, unsigned int *parity_critical_start,
		  unsigned int *parity_critical_end)
{
  switch (board_type) {
#if MX_2G_ENABLED
  case MX_BOARD_TYPE_D:
    *mcp = &mx_mcp_array_d[0];
    *len = mx_mcp_array_d_length;
    *parity_critical_start = mx_mcp_d_parity_critical_start;
    *parity_critical_end = mx_mcp_d_parity_critical_end;
    break;

  case MX_BOARD_TYPE_E:
    *mcp = &mx_mcp_array_e[0];
    *len = mx_mcp_array_e_length;
    *parity_critical_start = mx_mcp_e_parity_critical_start;
    *parity_critical_end = mx_mcp_e_parity_critical_end;
    break;
#endif
  
#if MX_10G_ENABLED
  case MX_BOARD_TYPE_Z:
    *mcp = &mx_mcp_array_z[0];
    *len = mx_mcp_array_z_length;
    *parity_critical_start = 0;
    *parity_critical_end = 0;
    break;
#endif
  
  default:
    /* we should never get here */
    *mcp = NULL;
    *len = 0;
    break;
  }

  if (*len == 0)
    return 1;
  
  return 0;
}

int
mx_mcp_set_globals(int unit, volatile uint8_t *sram)
{
  volatile mx_mcp_public_global_t *mcp_globals =
    (volatile mx_mcp_public_global_t *)(sram + MX_MCP_GLOBAL_OFFSET);

  MX_PIO_WRITE(&(mcp_globals->mcp_version), htonl(Globals(unit).mcp_version));
  MX_PIO_WRITE(&(mcp_globals->driver_api_version), htonl(Globals(unit).driver_api_version));
  MX_PIO_WRITE(&(mcp_globals->nodes_cnt), htonl(Globals(unit).nodes_cnt));
  MX_PIO_WRITE(&(mcp_globals->endpoints_cnt), htonl(Globals(unit).endpoints_cnt));
  MX_PIO_WRITE(&(mcp_globals->send_handles_cnt), htonl(Globals(unit).send_handles_cnt));
  MX_PIO_WRITE(&(mcp_globals->pull_handles_cnt), htonl(Globals(unit).pull_handles_cnt));
  MX_PIO_WRITE(&(mcp_globals->push_handles_cnt), htonl(Globals(unit).push_handles_cnt));
  MX_PIO_WRITE(&(mcp_globals->rdma_windows_cnt), htonl(Globals(unit).rdma_windows_cnt));
  MX_PIO_WRITE(&(mcp_globals->peer_hash_size), htonl(Globals(unit).peer_hash_size));
  MX_PIO_WRITE(&(mcp_globals->intr_coal_delay), htonl(Globals(unit).intr_coal_delay));
  MX_PIO_WRITE(&(mcp_globals->host_intr_queue[0].low), 
	       htonl(Globals(unit).host_intr_queue[0].low));
  MX_PIO_WRITE(&(mcp_globals->host_intr_queue[0].high), 
	       htonl(Globals(unit).host_intr_queue[0].high));
  MX_PIO_WRITE(&(mcp_globals->host_intr_queue[1].low), 
	       htonl(Globals(unit).host_intr_queue[1].low));
  MX_PIO_WRITE(&(mcp_globals->host_intr_queue[1].high), 
	       htonl(Globals(unit).host_intr_queue[1].high));
  MX_PIO_WRITE(&(mcp_globals->mac_high32), htonl(Globals(unit).mac_high32));
  MX_PIO_WRITE(&(mcp_globals->mac_low16), htons(Globals(unit).mac_low16));
  MX_PIO_WRITE(&(mcp_globals->mac_high16), htons(Globals(unit).mac_high16));
  MX_PIO_WRITE(&(mcp_globals->mac_low32), htonl(Globals(unit).mac_low32));
  MX_PIO_WRITE(&(mcp_globals->random_seed), htonl(Globals(unit).random_seed));
  MX_PIO_WRITE(&(mcp_globals->raw_recv_enabled), 0);
  MX_PIO_WRITE(&(mcp_globals->mapping_in_progress), 0);
  MX_STBAR();
  return(0);
}

#define MX_WRAPPER_SETVAL(str, param, unit, name) do {			\
  if (!strcmp(str, param)) {						\
    Globals(unit).name = val;						\
    if (sram) {								\
      MX_PIO_WRITE(&(mcp_globals->name), htonl(Globals(unit).name));	\
      MX_STBAR();							\
    }									\
    return (0);								\
  }									\
} while(0)

#define MX_WRAPPER_GETVAL(str, param, unit, name) do {			\
  if (!strcmp(str, param )) {						\
    if (sram)								\
      Globals(unit).name = ntohl(MX_PIO_READ(&(mcp_globals->name)));	\
    *val = Globals(unit).name;						\
    return (0);								\
  }									\
} while(0)


int
mx_mcp_set_param(int unit, volatile uint8_t *sram, const char *param, 
		 uint32_t val)
{
  volatile mx_mcp_public_global_t *mcp_globals = 
    (volatile mx_mcp_public_global_t *) (sram + MX_MCP_GLOBAL_OFFSET);
  
  MX_WRAPPER_SETVAL("parity_status", param, unit, parity_status);
  MX_WRAPPER_SETVAL("driver_api_version", param, unit, 
		    driver_api_version);
  MX_WRAPPER_SETVAL("host_intr_queue[0].low",  param, unit, 
		    host_intr_queue[0].low);
  MX_WRAPPER_SETVAL("host_intr_queue[1].low",  param, unit, 
		    host_intr_queue[1].low);
  MX_WRAPPER_SETVAL("host_intr_queue[0].high",  param, unit, 
		    host_intr_queue[0].high);
  MX_WRAPPER_SETVAL("host_intr_queue[1].high",  param, unit, 
		    host_intr_queue[1].high);

  MX_WRAPPER_SETVAL("raw_recv_enabled", param, unit, raw_recv_enabled);
  MX_WRAPPER_SETVAL("mapping", param, unit, mapping_in_progress);

  MX_WRAPPER_SETVAL("mac_high32", param, unit, mac_high32);
  MX_WRAPPER_SETVAL("mac_low16", param, unit, mac_low16);
  MX_WRAPPER_SETVAL("mac_high16", param, unit, mac_high16);
  MX_WRAPPER_SETVAL("mac_low32", param, unit, mac_low32);
  MX_WRAPPER_SETVAL("nodes_cnt", param, unit, nodes_cnt);
  MX_WRAPPER_SETVAL("endpoints_cnt", param, unit, endpoints_cnt);
  MX_WRAPPER_SETVAL("send_handles_cnt", param, unit, send_handles_cnt);
  MX_WRAPPER_SETVAL("pull_handles_cnt", param, unit, pull_handles_cnt);
  MX_WRAPPER_SETVAL("push_handles_cnt", param, unit, push_handles_cnt);
  MX_WRAPPER_SETVAL("rdma_windows cnt", param, unit, rdma_windows_cnt);

  MX_WRAPPER_SETVAL("random_seed", param, unit, random_seed);
  MX_WRAPPER_SETVAL("ethernet_mtu", param, unit, ethernet_mtu);
  MX_WRAPPER_SETVAL("ethernet_smallbuf", param, unit, ethernet_smallbuf);
  MX_WRAPPER_SETVAL("ethernet_bigbuf", param, unit, ethernet_bigbuf);

  MX_WRAPPER_SETVAL("local_peer_index", param, unit, local_peer_index);
  MX_WRAPPER_SETVAL("intr_coal_delay", param, unit, intr_coal_delay);
  MX_WRAPPER_SETVAL("endpt_recovery", param, unit, endpt_recovery);
  MX_WRAPPER_SETVAL("params_ready", param, unit,  params_ready);


  /* this is only hit if a string doesn't match above */
  return (ENOENT);
}

int
mx_mcp_get_param(int unit, volatile uint8_t *sram, const char *param, 
		 uint32_t *val)
{
  volatile mx_mcp_public_global_t *mcp_globals = 
    (volatile mx_mcp_public_global_t *) (sram + MX_MCP_GLOBAL_OFFSET);

  MX_WRAPPER_GETVAL("mcp_version", param, unit, mcp_version);
  MX_WRAPPER_GETVAL("mcp_status", param, unit, mcp_status);
  MX_WRAPPER_GETVAL("parity_status", param, unit, parity_status);
  MX_WRAPPER_GETVAL("reboot_status", param, unit, reboot_status);
  MX_WRAPPER_GETVAL("counters_offset", param, unit, counters_offset);
  MX_WRAPPER_GETVAL("logging_offset", param, unit, logging_offset);
  MX_WRAPPER_GETVAL("mac_high32", param, unit, mac_high32);
  MX_WRAPPER_GETVAL("mac_low16", param, unit, mac_low16);
  MX_WRAPPER_GETVAL("mac_high16", param, unit, mac_high16);
  MX_WRAPPER_GETVAL("mac_low32", param, unit, mac_low32);
  MX_WRAPPER_GETVAL("nodes_cnt", param, unit, nodes_cnt);
  MX_WRAPPER_GETVAL("kreqq_offset", param, unit, kreqq_offset);
  MX_WRAPPER_GETVAL("print_buffer_addr", param, unit, print_buffer_addr);
  MX_WRAPPER_GETVAL("print_buffer_pos", param, unit, print_buffer_pos);
  MX_WRAPPER_GETVAL("clock_freq", param, unit, clock_freq);
  MX_WRAPPER_GETVAL("intr_coal_delay", param, unit, intr_coal_delay);  
  MX_WRAPPER_GETVAL("endpt_recovery", param, unit, endpt_recovery);
  MX_WRAPPER_GETVAL("params_ready", param, unit,  params_ready);

  /* these are somewhat special, as they involve fetching read-only
     data which is not contained in the globals structure itself */
     
  if (!strcmp("print_limit_addr", param)) {
    *val = 
      MX_MCP_GLOBAL_OFFSET + 
      (uint32_t)offsetof(mx_mcp_public_global_t, print_buffer_limit);
    return 0;
  }

  if (!strcmp("Command queue offset", param)) {
    *val = 
      MX_MCP_GLOBAL_OFFSET + (uint32_t) offsetof(mx_mcp_public_global_t, 
						 command_queue);
    return 0;
  }

  if (!strcmp("MX_MCP_INTRQ_SLOTS", param)) {
    *val = MX_MCP_INTRQ_SLOTS;
    return 0;
  }

  if (!strcmp("MX_MCP_COMMANDQ_SLOTS", param)) {
    *val = MX_MCP_COMMANDQ_SLOTS;
    return 0;
  }

  if (!strcmp("MX_MCP_KREQQ_CNT", param)) {
    *val = MX_MCP_KREQQ_CNT;
    return 0;
  }

  if (!strcmp("raw_recv_vpages", param)) {
    *val = MX_MCP_GLOBAL_OFFSET 
      + (uint32_t)offsetof(mx_mcp_public_global_t, raw_recv_vpages);
    return 0;
  }

  if (!strcmp("raw_host_recv_offset", param)) {
    *val = MX_MCP_GLOBAL_OFFSET 
      + (uint32_t)offsetof(mx_mcp_public_global_t, raw_host_recv_offset);
    return 0;
  }

  if (!strcmp("host_query_vpage", param)) {
    *val = MX_MCP_GLOBAL_OFFSET 
      + (uint32_t)offsetof(mx_mcp_public_global_t, host_query_vpage);
    return 0;
  }

  if (!strcmp("MX_MCP_SENDQ_VPAGE_CNT", param)) {
    *val = MX_MCP_SENDQ_VPAGE_CNT;
    return 0;
  }
  
  if (!strcmp("MX_MCP_RECVQ_VPAGE_CNT", param)) {
    *val = MX_MCP_RECVQ_VPAGE_CNT;
    return 0;
  }
  
  if (!strcmp("MX_MCP_EVENTQ_VPAGE_CNT", param)) {
    *val = MX_MCP_EVENTQ_VPAGE_CNT;
    return 0;
  }

  if (!strcmp("MX_MCP_UMMAP_SIZE", param)) {
    *val = MX_MCP_UMMAP_SIZE;
    return 0;
  }

  if (!strcmp("MX_MCP_UDATAQ_SIZE", param)) {
    *val = MX_MCP_UDATAQ_SIZE;
    return 0;
  }

  if (!strcmp("MX_MCP_UREQQ_CNT", param)) {
    *val = MX_MCP_UREQQ_CNT * sizeof (mcp_ureq_t);
    return 0;
  }

  if (!strcmp("MX_MCP_ROUTE_BLOCK_SIZE", param)) {
    *val = MX_MCP_ROUTE_CNT_BLOCK * (MX_MCP_ROUTE_MAX_LENGTH + 1);
    return 0;
  }

  if (!strcmp("MX_MCP_PRINT_BUFFER_SIZE", param)) {
    *val = MX_MCP_PRINT_BUFFER_SIZE;
    return 0;
  }

  if (!strcmp("hostname_len", param)) {
    *val = (uint32_t)sizeof(mcp_globals->hostname);
    return 0;
  }

  if (!strcmp("hostname_addr", param)) {
    *val = MX_MCP_GLOBAL_OFFSET + (uint32_t)offsetof(mx_mcp_public_global_t, 
						     hostname);
    return 0;
  }

  if (!strcmp("ethernet_tx_ring", param)) {
    *val = MX_MCP_GLOBAL_OFFSET + (uint32_t)offsetof(mx_mcp_public_global_t, 
						     ether_tx_ring);
    return 0;
  }

  if (!strcmp("ethernet_tx_cnt", param)) {
    *val = MX_MCP_GLOBAL_OFFSET + (uint32_t)offsetof(mx_mcp_public_global_t, 
						     ether_tx_cnt);
    return 0;
  }

  if (!strcmp("ethernet_rx_small_ring", param)) {
    *val = MX_MCP_GLOBAL_OFFSET + (uint32_t)offsetof(mx_mcp_public_global_t, 
						     ether_rx_small_ring);
    return 0;
  }

  if (!strcmp("ethernet_rx_small_cnt", param)) {
    *val = MX_MCP_GLOBAL_OFFSET + (uint32_t)offsetof(mx_mcp_public_global_t, 
						     ether_rx_small_cnt);
    return 0;
  }

  if (!strcmp("ethernet_rx_big_ring", param)) {
    *val = MX_MCP_GLOBAL_OFFSET + (uint32_t)offsetof(mx_mcp_public_global_t, 
						     ether_rx_big_ring);
    return 0;
  }

  if (!strcmp("ethernet_rx_big_cnt", param)) {
    *val = MX_MCP_GLOBAL_OFFSET + (uint32_t)offsetof(mx_mcp_public_global_t, 
						     ether_rx_big_cnt);
    return 0;
  }

  if (!strcmp("MCP uptime offset", param)) {
    /* uptime is always the first counter */
    *val = ntohl(MX_PIO_READ(&(mcp_globals->counters_offset)));
    return 0;
  }
  
  /* this is only hit if a string doesn't match above */
  return (ENOENT);
}

